import * as THREE from 'https://unpkg.com/three/build/three.module.js';

/** Setup tooltip for element */
function setupTooltip( element, template ) {

	let tooltip = element.tooltipContent ?? '';
	if( typeof template !== 'undefined' && template !== '{tooltipContent}' && template.length > 0 ){

		tooltip = template.replace(/\{([^}]+)\}/g, function(match, key) {
			// Split the key by dot notation
			const keys = key.split('.');
			let value = element;

			// Traverse the object using the keys
			for (let i = 0; i < keys.length; i++) {
				value = value ? value[keys[i]] : undefined;
			}

			// If the value is undefined, replace it with an empty string or handle as necessary
			return value !== undefined ? value : '';
		});
	}

	if ( tooltip !== '') {
		tooltip = '<div class="itt_globe_' + element.globe_id + ' itt_globe_tooltip">' + tooltip + '</div>';
	}
	return tooltip;
}

/**
 * Prepares safe url
 * @param string str url to prepare
 * @returns string prepared URL
 */
const globe_ut_prepareURL = function (str) {

	if (typeof str !== "string") {
		return str;
	}

	// parse html entities
	str = str.replace(/&amp;/gi, '&');
	str.replace(/&#(\d+);/g, function (match, dec) {
		return String.fromCharCode(dec);
	});

	// check if allowed url
	var url, protocols;
	try {
		url = new URL(str);
	} catch (_) {
		url = false;
	}

	// check if valid url. If string it might be a anchor or relative path, so keep going
	if (url) {
		// acepted protocols
		protocols = [null, "http:", "https:", "mailto:", "tel:"];
		if (!protocols.includes(url.protocol)) {
			console.log('URL protocol not allowed');
			return '';
		}
	}

	return str;

};

/** Prepare data to be used. Set defaults and other operations */
const globe_ut_prepareData = function ( data ) {

	if ( ! data.hasOwnProperty( 'dotLabels' ) || ! data.dotLabels ) {
		data['dotlabels'] = [];
	}
	if ( ! data.hasOwnProperty( 'points' ) || ! data.points ) {
		data['points'] = [];
	}

	// set preview defaults
	let defaults = {
		'showGraticules': false,
		'backgroundColor': 'transparent',
		'atmosphere': {
			'enabled': false,
			'atmosphereColor': null,
			'atmosphereAltitude': null
		},
		'globeImage': 'earth-day.jpg'
	};

	// merge data with defaults
	if ( Object.keys(data).length === 0 || data.emptyPreview ) {
		Object.assign(data, defaults);
	}



	// set point of view
	data.pointOfView = {
		lat: 0,
		lng: 0,
		altitude: 1.5
	};

	// shouldn't be empty
	if (data.centerCoordinates) {
		let altitude = typeof data.altitudeOptions !== 'undefined' ? data.altitudeOptions.altitude : data.altitude;
		data.pointOfView.lat      = globe_ut_float(data.centerCoordinates.latitude);
		data.pointOfView.lng      = globe_ut_float(data.centerCoordinates.longitude);
		data.pointOfView.altitude = globe_ut_float(altitude);
	}

	return data;
};

const globe_ut_prepareProData = function ( data ) {

	if ( ! data.hasOwnProperty( 'arcLines' ) || ! data.arcLines ) {
		data['arcLines'] = [];
	}
	if ( ! data.hasOwnProperty( 'regions' ) || ! data.arcLines ) {
		data['regions'] = [];
	}
	if ( ! data.hasOwnProperty( 'html' ) || ! data.html ) {
		data['html'] = [];
	}

	return data;
};


const globe_ut_setupData = function ( data, dataTypes ) {
	for (const key in dataTypes) {
		let defaults = dataTypes[key];
		if ( data[key] && Array.isArray( data[key] ) ) {
			data[key].map(function (entry) {
				// add defaults data type
				if ( typeof entry.useCustom === "undefined" ||
					( typeof entry.useCustom !== "undefined" && ! globe_ut_bool( entry.useCustom ) ) ) {
					Object.assign(entry, data[defaults]);
				}

				if(Array.isArray(entry.action)){
					entry.action = entry.action[0];
				}

				// if it's using default action
				if ( data[defaults] && entry.action && entry.action === 'default') {
					entry.action = data[defaults]['action'];
				}
			});
		}
	}
};

/**
 * Prepare viewport and globe
 * @param DOM element container
 * @param object data
 * @param Globe Object thisGlobe
 */
const globe_ut_prepareViewport = function ( container, data ) {
	var aspRatioContainer = container.closest(".itt_globe_aspect_ratio");

	// container size adjustment
	// if mobile
	if (window.innerWidth <= 780 && typeof data.paddingTop !== 'undefined' && data.paddingTop !== '') {
		aspRatioContainer.style.paddingTop = String(data.paddingTopMobile) + '%';
	} else {
		aspRatioContainer.style.paddingTop = String(data.paddingTop) + '%';
	}
};

/**
 * Returns float, even if not a number
 * @param {string} string to get float from
 * @returns int float number
 */
const globe_ut_float = function (string) {
	var float = parseFloat(string);
	if (isNaN(float)) {
		return 0;
	}
	return float;
};

/**
 * Returns boolean value of a string or number
 * @param {*} string
 * @returns boolen
 */
const globe_ut_bool = function (string) {
	var bool = Number(string) === 0 || string === "false" || typeof string === "undefined" ? false : true;
	return bool;
};

const globe_ut_prepare_coordinates = function( type, marker ){

	if( typeof marker.coordinates !== 'undefined' && marker.coordinates[type] ){
		return parseFloat( marker.coordinates[type] );
	}

	if( typeof marker[type] !== 'undefined' ){
		return parseFloat( marker[type] );
	}
	return false;
};

/** Setup click events for element */
function setupClickEvent( thisGlobe, GlobeData, element ) {

	let layerDataTypes = {
		'arcLines': 'arcLineDefaults',
		'points': 'pointDefaults',
		'dotLabels': 'labelDefaults',
		'regions': 'regionDefaults',
		'html': 'htmlDefaults',
	};

	if(element.type === 'regions'){
		element = { ...element.properties };
	}

	if (new URLSearchParams(window.location.search).has('debug')) {
		console.debug(element);
	}

	// if we arrive here and it's still set as default, we need to set it to the default action for that layer
	if( element.action === 'default' ){
		element.action = GlobeData[layerDataTypes[element.type]].action;
	}

	if (element.action === "none") {
		// do nothing
		return;
	}

	const clickAction = element.action;
	if (clickAction && typeof window.ittGlobes.clickActions[clickAction] === 'function') {
		window.ittGlobes.clickActions[clickAction]( thisGlobe, GlobeData, element );
	}
	if ( ittGlobeData.isAdmin ) {
		return;
	}

	// check urls
	if (element.action === "open_url" || element.action === "open_url_new") {
		element.content = globe_ut_prepareURL(element.content);
	}
	// open new url
	if (element.action === "open_url" && element.content !== "") {
		document.location = element.content;
	} else if (element.action === "open_url_new" && element.content !== "") {
		window.open(element.content);
	}
}

/** Setup hover events for element */
function setupHoverEvent(thisGlobe, element, prevElement, type) {
	
	let GlobeData = thisGlobe.meta;
	
	if (type === 'regions') {
		thisGlobe.polygonCapColor(function (region) {
			// when hovering the globe itself, element doesn't exist or doesn't have properties
			if (
				typeof element === 'undefined' ||
				(element && typeof element.properties === 'undefined')
			) {
				return region.properties.color
					? region.properties.color
					: GlobeData.regionDefaults.inactive;
			}
			// change hover color if being hovered or part of same group
			else if (region === element) {
				return region.properties.hover;
			} else if (
				region.properties.originalId &&
				element &&
				element.properties &&
				element.properties.id &&
				region.properties.originalId.includes(element.properties.id)
			) {
				return region.properties.hover;
			} else if (
				region.properties.originalId &&
				element &&
				element.properties &&
				Array.isArray(element.properties.id) &&
				element.properties.id.includes(region.properties.id)
			) {
				return region.properties.hover;
			}
			// we actually don't need the previous I think, we can reset the value for the rest of the entries
			return region.properties.color
				? region.properties.color
				: GlobeData.regionDefaults.inactive;
		});
	} 

	else if (type === 'pin' || type === 'marker') {
		if (element) {
			// Convert hex color to integer
			const hoverColor = parseInt(element.hover.replace('#', '0x'), 16);
			parseInt(element.color.replace('#', '0x'), 16);

			let threeObj = element.__threeObjObject ? element.__threeObjObject : element.__threeObj;

			if (type === 'pin' ) {
				threeObj.children[0].children.forEach((element) => {
					element.material.color.setHex(hoverColor);
				});
			} else if (type === 'marker') {
				threeObj.children[0].material.color.setHex(hoverColor);
			}
		}

		if (prevElement && prevElement !== element) {
			const originalColor = parseInt(prevElement.color.replace('#', '0x'), 16);
			let threeObj = prevElement.__threeObjObject ? prevElement.__threeObjObject : prevElement.__threeObj;
			
			if (type === 'pin') {
				threeObj.children[0].children.forEach((element) => {
					element.material.color.setHex(originalColor);
				});
			} else if (type === 'marker' && prevElement.type === 'marker') {
				threeObj.children[0].material.color.setHex(originalColor);
			}
		}
		

		//pause/resume animation
		if (
			typeof GlobeData.rotate !== 'undefined' &&
			globe_ut_bool(GlobeData.rotate.enabled)
		) {
			thisGlobe.controls().autoRotate = !element;
		}

	} 
	
	else { // for default points and labels
		thisGlobe.labelColor(function (el) {
			return element && el.id === element.id ? el.hover : el.color;
		});

		thisGlobe.pointColor(function (el) {
			return element && el.id === element.id ? el.hover : el.color;
		});

		//pause/resume animation
		if (
			typeof GlobeData.rotate !== 'undefined' &&
			globe_ut_bool(GlobeData.rotate.enabled)
		) {
			thisGlobe.controls().autoRotate = !element;
		}
	}

	return thisGlobe;
}

/** Add Points Layer to Globe */
function addPointsLayer(thisGlobe, GlobeData) {
	thisGlobe.pointsData( GlobeData.points )
		.pointLat(p => globe_ut_prepare_coordinates('latitude', p) )
		.pointLng(p => globe_ut_prepare_coordinates('longitude', p) )
		.pointRadius(p => p.radius / 10)
		.pointAltitude(p => p.altitude / 100)
		.pointColor(p => p.color)
		.pointResolution(20);

	// tooltip
	thisGlobe.pointLabel(function (label) {

		let pointsTooltipTemplate = GlobeData.tooltipTemplate ?? '{tooltipContent}';
		if( GlobeData.pointsTooltipTemplate ){
			pointsTooltipTemplate = GlobeData.pointsTooltipTemplate;
		}
		return setupTooltip(label, pointsTooltipTemplate );

	});

	// click events
	thisGlobe.onPointClick(function ( element ) {
		setupClickEvent( thisGlobe, GlobeData, element );
	});

	// hover events
	thisGlobe.onPointHover(function (label) {
		setupHoverEvent( thisGlobe, label, 'point' );
	});

	// click events
	thisGlobe.onObjectClick(function ( element ) {
		setupClickEvent( thisGlobe, GlobeData, element );
	});
}

/** Add Labels Layer to Globe */
function addLabelsLayer( thisGlobe, GlobeData ) {

	thisGlobe.labelsData( GlobeData.dotLabels )
		.labelLat(l => globe_ut_prepare_coordinates('latitude', l))
		.labelLng(l => globe_ut_prepare_coordinates('longitude', l))
		.labelText(l => l.title ? (l.title.trim() ? l.title : '_') : '_')
		.labelAltitude(l => l.altitude / 100)
		.labelSize(l => l.size / 10)
		.labelColor(l => l.color)
		.labelResolution(20)
		.labelIncludeDot(l => globe_ut_bool(l.includeDot))
		.labelDotRadius(l => l.radius / 10)
		.labelDotOrientation(l => l.dotOrientation);

	// font
	if (GlobeData.labelFont && GlobeData.labelFont !== 'default') {

		let url = ittGlobeData.assetsUrl + 'fonts/' + GlobeData.labelFont + '.json';

		fetch(url)
			.then(res => res.json())
			.then( labelfont => { thisGlobe.labelTypeFace(labelfont); } )
			.catch(error => {
				console.error('Fetch error:', error);
			});
	}

	// tooltip
	thisGlobe.labelLabel(function (label) {

		let dotLabelsTooltipTemplate = GlobeData.tooltipTemplate ?? '{tooltipContent}';
		if( GlobeData.dotLabelsTooltipTemplate ){
			dotLabelsTooltipTemplate = GlobeData.dotLabelsTooltipTemplate;
		}
		return setupTooltip(label, dotLabelsTooltipTemplate );
	});

	// click events
	thisGlobe.onLabelClick(function (label, event, coordinates) {
		setupClickEvent( thisGlobe, GlobeData, label );
	});

	// hover events
	thisGlobe.onLabelHover(function (label) {
		setupHoverEvent( thisGlobe, label, 'label' );
	});
}

/** Setup click events for element */
function setupResizeEvent( thisGlobe, container, data ) {

	var aspRatioContainer = container.closest(".itt_globe_aspect_ratio");

	window.addEventListener('resize', function () {
		if (window.innerWidth <= 780 && typeof data.paddingTop !== 'undefined' && data.paddingTop !== '') {
			aspRatioContainer.style.paddingTop = String(data.paddingTopMobile) + '%';
		} else {
			aspRatioContainer.style.paddingTop = String(data.paddingTop) + '%';
		}

		// Here should go the code to rebuild globe with new size in a more performant way
		// check for size changes, etc..
		if (thisGlobe.width() !== container.offsetWidth) {
			thisGlobe
				.width(container.offsetWidth)
				.height(container.offsetHeight);
		}
	});
}

/**
 * Controls globe interactions
 * @param {*} thisGlobe
 * @param {*} data
 */
function setupInteractions( thisGlobe, data ) {
		// interactions
		if( typeof data.interactions !== 'undefined' ) {
			thisGlobe.controls().enableZoom   = globe_ut_bool( data.interactions.zoom );
			thisGlobe.controls().enablePan    = globe_ut_bool( data.interactions.pan );
			thisGlobe.controls().enableRotate = globe_ut_bool( data.interactions.pan );
		}

		if( typeof data.interactions === 'undefined' || globe_ut_bool( data.interactions.zoom ) ) {
						
			let lastClickTime = 0;
			const doubleClickDelay = 300; // milliseconds
			
			thisGlobe.onGlobeClick(function(coordinates, event) {

				const currentTime = new Date().getTime();
				const timeDiff = currentTime - lastClickTime;
				
				if (timeDiff < doubleClickDelay) {
					// Double click detected - zoom in

					let zoomTo = thisGlobe.pointOfView();
					zoomTo.altitude = zoomTo.altitude * 0.6;
					zoomTo.lat = coordinates.lat;
					zoomTo.lng = coordinates.lng;
					thisGlobe.pointOfView(zoomTo, 800);
				}
				
				lastClickTime = currentTime;
			});
		}
}

function setupZoomEvent(globe) {

    let GlobeData = globe.meta;

	globe.onZoom(function(coordinates) {
		if (GlobeData.altitudeOptions) {
			// Clamp altitude between min and max values if values are set
			const minAlt = parseFloat(GlobeData.altitudeOptions.minAltitude);
			const maxAlt = parseFloat(GlobeData.altitudeOptions.maxAltitude);
			
			if (!isNaN(minAlt) && minAlt !== 0 && coordinates.altitude < minAlt) {
				globe.pointOfView({
					...coordinates,
					altitude: minAlt
				});
			}
			else if (!isNaN(maxAlt) && maxAlt !== 0 && coordinates.altitude > maxAlt) {
				globe.pointOfView({
					...coordinates, 
					altitude: maxAlt
				});
			}
		}
	});
}

let ittGlobes$1 = {
	'globes': [],
	'globesIndex': {},
	'clickActions': {},
};

/** initialize globe */
ittGlobes$1.init = function () {
	// return early if Globe.gl Library didn't load properly
	if (typeof Globe === 'undefined') {
		console.error('Globe lib not loaded properly');
		return;
	}

	// check for globe container created by the shortcode
	let globe_render_container = document.getElementsByClassName( 'js-itt-globe-render' );
	for (const globe_container of globe_render_container) {
		if ( ! globe_container.dataset.hasOwnProperty('globe_id') ) {
			continue;
		}
		let globe_id = 0;
		try {
			globe_id = parseInt( JSON.parse(globe_container.dataset.globe_id ) );
		} catch (e) {
			continue;
		}

		if ( ittGlobes$1.globesIndex[ globe_id ] !== undefined ) {
			continue;
		}

		if ( ! globe_container.dataset.hasOwnProperty( 'globe_meta' ) ) {
			continue;
		}

		let globe_meta = {};
		try {
			globe_meta = JSON.parse( globe_container.dataset.globe_meta );
		} catch (e) {
			globe_meta = {};
		}
		create_globe( globe_id, globe_meta );
	}
};


var create_globe = function ( id, data ) {

	//data types - normal
	let dataTypes = {
		'points': 'pointDefaults',
		'dotLabels': 'labelDefaults',
	};
	let container = document.getElementById( 'itt_globe_' + id );

	globe_ut_prepareData( data );
	globe_ut_setupData( data, dataTypes );
	globe_ut_prepareViewport( container, data );

	let configOptions = {
		rendererConfig: { antialias: true, alpha: true },
		waitForGlobeReady: true,
		animateIn: typeof data.animateIn !== 'undefined' ? globe_ut_bool( data.animateIn ) : true,
	};

	let thisGlobe = Globe( configOptions )
		(document.getElementById( 'itt_globe_' + id ));

	// to make it publically accessible
	thisGlobe.meta = data;

	// background colour
	if (data.backgroundColor === 'transparent') {
		data.backgroundColor = 'rgba(0,0,0,0)';
	}

	// globe visuals
	if ( data.globeImage !== null && data.globeImage !== '' && data.globeImage !== 'earth-hollow' && data.globeImage !== 'noImage' && data.globeImage !== 'customImage' ) {
		thisGlobe.globeImageUrl( ittGlobeData.imagesUrl + data.globeImage );
	}

	thisGlobe.backgroundColor(data.backgroundColor)
		.width(container.offsetWidth)
		.height(container.offsetHeight)
		.showGraticules( globe_ut_bool(data.showGraticules))
		.showAtmosphere( globe_ut_bool(data.atmosphere.enabled))
		.atmosphereColor(data.atmosphere.atmosphereColor)
		.atmosphereAltitude(data.atmosphere.atmosphereAltitude)
		.pointOfView(data.pointOfView);

	setupInteractions( thisGlobe, data );
	setupResizeEvent( thisGlobe, container, data );
	setupZoomEvent( thisGlobe );
	// if admin
	if ( ittGlobeData.isAdmin ) {

		thisGlobe.onGlobeReady(function () {
			const event = new Event("adminGlobeReady");
			document.dispatchEvent(event);
		});
	}
	// setup points
	if ( data.points ) {
		addPointsLayer( thisGlobe, data );
	}

	// setup label points
	if ( data.dotLabels ) {
		addLabelsLayer( thisGlobe, data );
	}

	// add click event function to be accesible globally
	thisGlobe.setupClickEvent = setupClickEvent;
	thisGlobe.setupHoverEvent = setupHoverEvent;

	// add object to be public
	ittGlobes$1.globes.push( thisGlobe );
	ittGlobes$1.globesIndex[ id ] = thisGlobe;

	// altitude options
	if ( data.altitudeOptions ) {
		thisGlobe.altitudeOptions = data.altitudeOptions;
	}
};

ittGlobes$1.init();

window.ittGlobes = ittGlobes$1;

/** Add Arcs Layer to Globe - Pro */
function addArcsLayer( thisGlobe, data ) {

	const segmented_lines = processLinesArray(data.arcLines);

	thisGlobe.arcsData( segmented_lines )
		.arcStartLat(p => p.line[0].coordinates.latitude)
		.arcStartLng(p => p.line[0].coordinates.longitude)
		.arcEndLat(p => p.line[1].coordinates.latitude)
		.arcEndLng(p => p.line[1].coordinates.longitude)
		.arcColor( p => p.stroke )
		.arcStroke( p => p.strokeWidth )
		.arcDashLength( p => p.dashLength / 100 )
		.arcDashGap( p => p.dashGap / 100 )
		.arcDashAnimateTime( p => p.animate * 1000 )
		.arcAltitude(function (p) {
			if (isNaN(p.altitude) || p.altitude.trim() === '') {
				return null;
			}
			return p.altitude / 100;
		});
}

function processLinesArray(lines) {
	const segmented_lines = [];

	lines.forEach(entry => {
		const line_points = entry.line;

		if ( line_points.length <= 2) {
			segmented_lines.push(entry);
			return;
		}
		// create copy for all properties
		const arc_data = { ...entry };

		// replace the coordinates with a single line
		for (let i = 1; i < line_points.length; i++) {
			const new_arc_data = { ...arc_data };
			new_arc_data.line = [line_points[i - 1], line_points[i]];
			segmented_lines.push( new_arc_data );
		}
	});

	return segmented_lines;
}

/** Add Points Layer to Globe */
function addHTMLElementsLayer ( thisGlobe, GlobeData ) {

	thisGlobe.htmlElementsData( GlobeData.html )
		.htmlLat(p => globe_ut_prepare_coordinates('latitude', p) )
		.htmlLng(p => globe_ut_prepare_coordinates('longitude', p) )
		.htmlAltitude(0.01) // there was no benefit in using the altitude from the backend
        .htmlElement(p => {

            let html = p.html ? p.html : buildHtml(p, GlobeData);

            const el = document.createElement('div');
            el.innerHTML = html;
            el.classList.add('itt_globe_html_element');
            el.style['pointer-events'] = 'auto';
            el.onclick = function() {
                setupClickEvent(thisGlobe, GlobeData, p);
            };
            return el;
        });

}

function buildHtml(p, GlobeData) {
    let template = GlobeData.htmlDefaults.htmlTemplate;

    // replace placeholders with values from p
    template = template.replace(/\{([^}]+)\}/g, (match, key) => {
        // Handle dot notation by splitting on dots
        const keys = key.split('.');
        let value = p;
        
        // Traverse the object/array following the dot notation
        for (const k of keys) {
            if (value === undefined || value === null) return match;
            value = value[k];
        }

        return value !== undefined ? value : match;
    });

    return template;
}

/**
 * Adds geojson regions layer
 * @param {*} thisGlobe
 * @param {*} GlobeData
 */
const addRegionLayer = function( thisGlobe, GlobeData ) {

	// in case it's a hollow globe, we don't want to see the side color
	let sideColor = 'rgba(100, 100, 100, 0.05)';
	if( GlobeData.globeImage === 'earth-hollow'){
		sideColor = 'rgba(100, 100, 100, 0)';
	}

	if ( GlobeData.regionDefaults === undefined ) {
		GlobeData.regionDefaults = {};
	}

	let defaultStrokeColor = GlobeData.regionDefaults.strokeColor ?? '#f0f0f0';

	return new Promise((resolve) => {
		thisGlobe.polygonAltitude( r => r.properties.altitude ?? 0.005 )
			.lineHoverPrecision(0)
			.polygonCapColor(r => r.properties.color ? r.properties.color : GlobeData.regionDefaults.inactive)
			.polygonSideColor(() => sideColor)
			.polygonStrokeColor(r => r.properties.strokeColor || defaultStrokeColor)
			.polygonCapCurvatureResolution(5)
			.polygonsTransitionDuration(300);

		thisGlobe.regionLayers = {};
		thisGlobe.originalPolygonsData = [];

		// tooltip
		thisGlobe.polygonLabel(function (region) {

			// if region was not added manually and option to only display on active region is enabled, skip
			let onlyActive = globe_ut_bool(GlobeData.tooltipOnlyActive);
			if( onlyActive && typeof region.properties.originalId === 'undefined' ){
				return;
			}
			// set tooltip template
			let regionsTooltipTemplate = GlobeData.tooltipTemplate ?? '{tooltipContent}';
			if( GlobeData.regionsTooltipTemplate ){
				regionsTooltipTemplate = GlobeData.regionsTooltipTemplate;
			}

			return setupTooltip(region.properties, regionsTooltipTemplate);
		});

		// click events
		thisGlobe.onPolygonClick(function (region) {
			setupClickEvent(thisGlobe, GlobeData, region.properties);
		});

		// hover events
		thisGlobe.onPolygonHover(function (current, previous) {
			setupHoverEvent( thisGlobe, current, previous, 'regions' );
		});

		if (!GlobeData.regions || GlobeData.regions.length === 0) {
			resolve();
			//We can't return here, otherwise the empty regions will not load. If we add an option to hide emtpy regions, it would be here
		}

		// fetch regions
		(async () => {
			const regions = await prepareRegions(thisGlobe, GlobeData);
			thisGlobe.polygonsData(regions);
			thisGlobe.originalPolygonsData = regions;

			resolve();
		})();
	});
};

const prepareRegions = async function( thisGlobe, GlobeData, altitudeMultiplier ){

	// if it already exists, return it, otherwise fetch it
	if (thisGlobe.regionLayers && thisGlobe.regionLayers[GlobeData.id]) {
		return thisGlobe.regionLayers[GlobeData.id];
	}

	altitudeMultiplier = altitudeMultiplier ?? 0;

	let geojson_source = GlobeData.regionSource;
	if ( geojson_source === 'disabled' || geojson_source === undefined ){
		return;
	}
	if ( GlobeData.regions === undefined ){
		return;
	}

	let fullpath;
	if( geojson_source !== 'custom' ){
		if ( ! geojson_source.includes('.geojson') && ! geojson_source.includes('.json') ) {
			geojson_source += '.json';
		}
		fullpath = ittGlobeData.assetsUrl + 'geojson/' + geojson_source;
	} else {
		fullpath = GlobeData.regionCustomSourceURL;
	}

	const selectedRegionsIndividually = GlobeData.regions.reduce((acc, region) => {

		// save original id, to use for grouped hover
		region.originalId = region.id;

		// Destructure the region to get all properties except ids
		const { id, ...regionProps } = region;
		if ( ! id ) {
			return acc;
		}

		// Convert string id to array if needed
		if (typeof id === 'string') {
			id = id.includes(',') ? id.split(',').map(i => i.trim()) : [id];
		}

		// Create new objects with all region properties and the individual id
		const idsWithProps = id.map(id => ({ ...regionProps, id }));
		return acc.concat(idsWithProps);
	}, []);

	// Convert the selection of regions array to a dictionary for quick lookup
	const regionsDict = selectedRegionsIndividually.reduce((acc, item) => {
		// Add both id and name as keys if available
		acc[item.id] = item; // id from the regions object
		if (item.title) {     // title or name for country
			acc[item.title] = item;  // Store the title (e.g., "Algeria")
		}
		return acc;
	}, {});

	return fetch(fullpath)
		.then(res => res.json())
		.then(regionsAvailable => {
			// Enrich the GeoJSON data with each regions data
			regionsAvailable.features.forEach(feature => {
				const regionId = feature.id;
				const regionName = feature.properties.name;

				// Try to find the matching region data by id or name
				const regionData = regionsDict[regionId] || regionsDict[regionName];

				if (regionData) {
					feature.properties = {
						...feature.properties,
						...regionData
					};
				}
				// Add globe_id to feature properties
				feature.properties.globe_id = GlobeData.id;

				if (thisGlobe.liveFilter && thisGlobe.liveFilter.allLabel && thisGlobe.liveFilter.allLabel.trim() !== '') {
					feature.properties.altitude = 0.005 + ( 0.0025 * altitudeMultiplier );
				}

			});

			let features = regionsAvailable.features;

			// save total of available features
			thisGlobe.totalAvailableRegions = features.length;
			// get total of active regions
			thisGlobe.totalActiveRegions = features.filter(feature => feature.properties.originalId).length;

			// Filter to only active regions if setting enabled
			if (globe_ut_bool(GlobeData.onlyActiveRegions)) {
				features = features.filter(feature => feature.properties.originalId);
			}

			thisGlobe.regionLayers[GlobeData.id] = features;
			return features;
		});
};

/** Add Marker Objects Layer to Globe 
 * It will inherit the same properties as the points layer
*/
function addObjectsLayer ( thisGlobe, GlobeData ) {

	// use in images if needed
	thisGlobe.textureLoader = new THREE.TextureLoader();

	thisGlobe.objectsData( GlobeData.objects )
		.objectLat(p => globe_ut_prepare_coordinates('latitude', p) )
		.objectLng(p => globe_ut_prepare_coordinates('longitude', p) )
		.objectAltitude(p => 0);
	// object shape
	thisGlobe.objectThreeObject(function(d) {
		if (d.type === 'pin') {
			return createPin(d, thisGlobe);
		}
		if (d.type === 'marker') {
			return createMarker(d);
		}
		if (d.type === 'image') {
			return createImage(d, thisGlobe);
		}
		if (d.type === 'flag') {
			if(d.image?.url){
				return createFlagImage(d, thisGlobe);
			}
			else {
				return createFlag(d, thisGlobe);
			}
		}
		return false;
	});

	// tooltip
	thisGlobe.objectLabel(function (label) {

		let pointsTooltipTemplate = GlobeData.tooltipTemplate ?? '{tooltipContent}';
		if( GlobeData.pointsTooltipTemplate ){
			pointsTooltipTemplate = GlobeData.pointsTooltipTemplate;
		}
		return setupTooltip(label, pointsTooltipTemplate );

	});

	// click events
	thisGlobe.onObjectClick(function ( element ) {
		setupClickEvent( thisGlobe, GlobeData, element );
	});

	// hover events	
	thisGlobe.onObjectHover(function (object, prevObject) {
		let type = object && object.type ? object.type : prevObject.type;
		setupHoverEvent(thisGlobe, object, prevObject, type );
	});
}

function createFlag(d, thisGlobe) {
	const flagGroup = new THREE.Group();
	let globeRadius = thisGlobe.getGlobeRadius();
	let size = d.radius * 0.5;
	let height = size * (3/5);
	// Set minimum altitude without dividing by globe radius
	let minAltitude = height + 1;
	let altitude = Math.max(d.altitude, minAltitude);
	let altitudeInUnits = parseFloat((altitude * globeRadius).toFixed(2)) * 0.01;
	let poleRadius = size * 0.025;
	let animate = d.animate ? globe_ut_bool(d.animate) : true;
	let width = size;
	let flagMaterial;
	let flagPole = createFlagPole(d, thisGlobe);

	flagGroup.add(flagPole);

	flagMaterial = new THREE.MeshLambertMaterial({
		color: d.color,
		side: THREE.DoubleSide
	});
	
	const geometry = new THREE.PlaneGeometry(width, height, width, height);
	const flagMesh = new THREE.Mesh(geometry, flagMaterial);

	// Position flag mesh with bottom edge at the top of the pole
	flagMesh.position.set(width/2 + poleRadius, altitudeInUnits - height/2, 0);

	const animateFunction = () => {
		// Get the position attribute
		const positions = flagMesh.geometry.getAttribute('position');
		
		for (let y = 0; y <= height; y++) {
			for (let x = 0; x <= width; x++) {
				const index = x + y * (width + 1);
				// Create wave effect by combining sine waves
				const waveX = Math.sin(x * 0.5 - Date.now() * 0.003) * 0.5;
				const waveY = Math.sin(y * 0.3 - Date.now() * 0.004) * 0.3;
				const wave = waveX + waveY;
				// Increase amplitude towards flag edge
				const amplitude = (x / width) * 2;
				positions.array[index * 3 + 2] = wave * amplitude;
			}
		}
		
		// Mark the positions as needing an update
		positions.needsUpdate = true;
		requestAnimationFrame(animateFunction);
	};

	if(animate){
		animateFunction();
	}

	flagGroup.add(flagMesh);

	// Rotate entire flag group to be perpendicular to globe surface
	flagGroup.rotation.x = Math.PI / 2;
	return flagGroup;
}

function createFlagImage(d, thisGlobe) {
	const flagGroup = new THREE.Group();
	let globeRadius = thisGlobe.getGlobeRadius();
	let size = d.radius * 0.5;
	let height = size * (3/5);
	// Set minimum altitude without dividing by globe radius
	let minAltitude = height + 1;
	let altitude = Math.max(d.altitude, minAltitude);
	let altitudeInUnits = parseFloat((altitude * globeRadius).toFixed(2)) * 0.01;    
	let poleRadius = size * 0.025;
	let animate = d.animate ? globe_ut_bool(d.animate) : true;
	let width = size;
	let flagMaterial;
	let flagPole = createFlagPole(d, thisGlobe);
	
	flagGroup.add(flagPole);

	flagMaterial = new THREE.MeshBasicMaterial({
		map: thisGlobe.textureLoader.load(d.image.url),
		side: THREE.DoubleSide,
		transparent: true
	});
	
	const geometry = new THREE.PlaneGeometry(width, height, width, height);
	const flagMesh = new THREE.Mesh(geometry, flagMaterial);

	// Position flag mesh with bottom edge at the top of the pole
	flagMesh.position.set(width/2 + poleRadius, altitudeInUnits - height/2, 0);

	// Update geometry and position once image loads
	const img = new Image();
	img.onload = function() {
		const aspectRatio = img.height / img.width;
		const adjustedHeight = width * aspectRatio;
		
		// Update geometry with correct dimensions
		flagMesh.geometry = new THREE.PlaneGeometry(width, adjustedHeight, width, adjustedHeight);
		
		// Update position to account for new height, keeping bottom edge at top of pole
		flagMesh.position.set(width/2 + poleRadius, altitudeInUnits - adjustedHeight/2, 0);
	};
	img.src = d.image.url;

	const animateFunction = () => {
		// Get the position attribute
		const positions = flagMesh.geometry.getAttribute('position');
		
		for (let y = 0; y <= height; y++) {
			for (let x = 0; x <= width; x++) {
				const index = x + y * (width + 1);
				// Create wave effect by combining sine waves
				const waveX = Math.sin(x * 0.5 - Date.now() * 0.003) * 0.5;
				const waveY = Math.sin(y * 0.3 - Date.now() * 0.004) * 0.3;
				const wave = waveX + waveY;
				// Increase amplitude towards flag edge
				const amplitude = (x / width) * 2;
				positions.array[index * 3 + 2] = wave * amplitude;
			}
		}
		
		// Mark the positions as needing an update
		positions.needsUpdate = true;
		requestAnimationFrame(animateFunction);
	};

	if(animate){
		animateFunction();
	}

	flagGroup.add(flagMesh);

	// Rotate entire flag group to be perpendicular to globe surface
	flagGroup.rotation.x = Math.PI / 2;
	return flagGroup;
}

function createFlagPole(d, thisGlobe) {
	let globeRadius = thisGlobe.getGlobeRadius();
	let size = d.radius * 0.5;
	let height = size * (3/5);
	// Use the same minimum altitude calculation as the flag
	let minAltitude = height + 1;
	let altitude = Math.max(d.altitude, minAltitude);
	let altitudeInUnits = parseFloat((altitude * globeRadius).toFixed(2)) * 0.01;
	let poleRadius = size * 0.025;

	let geometry = new THREE.CylinderGeometry(poleRadius, poleRadius, altitudeInUnits, 16, 1);
	let material = new THREE.MeshPhongMaterial({
		color: d.color,
		specular: "#999999",
		shininess: 30
	});
	const pole = new THREE.Mesh(geometry, material);
	
	// Position the pole so its base is at the surface and it extends upward
	pole.position.set(0, altitudeInUnits/2, 0);

	return pole;
}

/** Create a 3d image object */
function createImage(d, thisGlobe) {

	const textureLoader = thisGlobe.textureLoader;
	const group = new THREE.Group();
	let size = d.radius;

	// currently not being used, could be a future feature
	let includeLine = d.includeLine ? globe_ut_bool(d.includeLine) : false;
	let animate = d.animate ? globe_ut_bool(d.animate) : false;

	// Get globe width and altitude in units
	const globeRadius = thisGlobe.getGlobeRadius();
	const altitudeInUnits = parseFloat((d.altitude * globeRadius).toFixed(2)) * 0.01;

	// Create a plane geometry for the image
	// Load the image and calculate its aspect ratio
	const img = new Image();
	img.onload = function() {
		const aspectRatio = img.height / img.width;
		const width = size;
		const height = width * aspectRatio;

		// Update plane geometry with calculated dimensions
		const planeGeometry = new THREE.PlaneGeometry(width, height);
		imagePlane.geometry = planeGeometry;

		// Ensure the plane is centered
		if(d.imageOrientation === 'horizontal'){
			imagePlane.position.set(0, 0, altitudeInUnits);
		} else {
			imagePlane.position.set(0, 0, altitudeInUnits + height / 2);
		}
		
	};
	
	img.src = d.image.url;
	
	const planeGeometry = new THREE.PlaneGeometry(size, size);

	// Load the texture from the image URL
	const texture = textureLoader.load(d.image.url);
	const planeMaterial = new THREE.MeshBasicMaterial({
		map: texture,
		transparent: true,
		side: THREE.DoubleSide
	});

	const imagePlane = new THREE.Mesh(planeGeometry, planeMaterial);

	// Position the image plane
	imagePlane.position.set(0, 0, altitudeInUnits);

	let orientation = new THREE.Vector3(0, -1, altitudeInUnits);
	if(d.imageOrientation === 'horizontal'){
		orientation = new THREE.Vector3(0, 0, 0);
	}

	// Rotate the image plane to face outward or vertical from the globe
	imagePlane.lookAt(orientation);

	group.add(imagePlane);
	
	// Create a line from the surface to the image
	// currently not being used, could be a future feature
	if (altitudeInUnits > 1 && includeLine) {
	const lineGeometry = new THREE.BufferGeometry().setFromPoints([
				new THREE.Vector3(0, 0, 0),  // Start from the globe surface
				new THREE.Vector3(0, 0, altitudeInUnits)  // End at the image
			]);
			
		const lineMaterial = new THREE.LineBasicMaterial({ color: d.color });
		const line = new THREE.Line(lineGeometry, lineMaterial);
		group.add(line);
	}

	// Add animation for vertical orientation
	// currently not being used, could be a future feature
	if (animate) {
		const animate = () => {
		imagePlane.rotation.y += 0.05; // Adjust rotation speed as needed
		requestAnimationFrame(animate);
		};
		animate();
	}

	return group;
       
}

/** Create a 3d marker object */
function createMarker(d, thisGlobe) {
	// Create the outer shape of the pin
	const outerShape = new THREE.Shape();
	outerShape.moveTo(0, 0);
	outerShape.bezierCurveTo(0, 0.8, 0.8, 1.6, 1.6, 1.6);
	outerShape.bezierCurveTo(2.4, 1.6, 3.2, 0.8, 3.2, 0);
	outerShape.bezierCurveTo(3.2, -0.8, 2.4, -2.4, 1.6, -4);
	outerShape.bezierCurveTo(0.8, -2.4, 0, -0.8, 0, 0);

	// Create the inner shape for the hole with a bigger radius and moved down
	const innerShape = new THREE.Path();
	innerShape.moveTo(1.6, 0.6);
	innerShape.absarc(1.6, 0.6, 0.9, 0, Math.PI * 2, false);

	outerShape.holes.push(innerShape);

	// Create extrusion settings
	const extrudeSettings = {
		steps: 2,
		depth: 0.4,
		bevelEnabled: true,
		bevelThickness: 0.2,
		bevelSize: 0.1,
		bevelSegments: 1
	};

	// Create geometry
	const geometry = new THREE.ExtrudeGeometry(outerShape, extrudeSettings);

	// Center the geometry so that its local origin (0, 0, 0) is in the middle
	geometry.center();

	// Create material
	const material = new THREE.MeshPhongMaterial({
		color: d.color,
		shininess: 100
	});

	// Create mesh
	const pinMesh = new THREE.Mesh(geometry, material);

	// Rotate the pin to face the globe
	pinMesh.lookAt(new THREE.Vector3(0, 0, 0));
	pinMesh.rotateX(Math.PI / 2);

	// Scale the pin
	const scale = d.radius * 0.1; // Reduced scale to make pins smaller
	pinMesh.scale.set(scale, scale, scale);

	// Move the pin up/away from the globe
	const offset = scale * 3 + d.altitude * 0.1; // Adjust this value to fine-tune the offset
	pinMesh.position.set(0, 0, offset);

	// Add animation for vertical orientation
	if (d.animate) {
		const animate = () => {
			pinMesh.rotation.y += 0.05; // Adjust rotation speed as needed
			requestAnimationFrame(animate);
		};
		animate();
	}

	return pinMesh;
}


/** Create a pin object */
function createPin(d, thisGlobe) {

	const group = new THREE.Group();

	// radius will originally be 20 for example. 
	// altitude will originally be 10 for example. 
	let AdjustedRadius = parseFloat((d.radius * 0.2).toFixed(2));
	let AdjustedAltitude = parseFloat((d.altitude / 100 ).toFixed(2));
	// Get globe radius and altitude in units
	const globeRadius = thisGlobe.getGlobeRadius();
	const altitudeInUnits = parseFloat((AdjustedAltitude * globeRadius).toFixed(2));
	// Sphere for the round part
	const sphereGeometry = new THREE.SphereGeometry(AdjustedRadius, 32, 32);
	const sphereMaterial = new THREE.MeshLambertMaterial({ color: d.color });
	const sphere = new THREE.Mesh(sphereGeometry, sphereMaterial);
	
	// Position the sphere so that it sits above the cone
	sphere.position.set(0, 0, altitudeInUnits);  // Adjusted position based on cone's rotation

	group.add(sphere);

	// Cone for the pointer
	const coneHeight = Math.max(0, parseInt(altitudeInUnits));
	if(coneHeight > 0){
	  const coneGeometry = new THREE.ConeGeometry(AdjustedRadius/4, coneHeight, 32);
	  const coneMaterial = new THREE.MeshLambertMaterial({ color: d.color });
	  const cone = new THREE.Mesh(coneGeometry, coneMaterial);
	 
	  cone.position.set(0, 0, coneHeight/2);
	  cone.rotation.y = -Math.PI / 2;  // Keep the rotation around Y
	  cone.rotation.x = -Math.PI / 2;  // Keep the rotation around X
	  
	  group.add(cone);
	}
	return group;
}

/**
 * Adds Auto Rotate and its speed controls
 * @param {*} thisGlobe
 * @param {*} data
 */
function addAutoRotate( thisGlobe, data ) {
	if (typeof data.rotate !== 'undefined' && globe_ut_bool(data.rotate.enabled)) {
		thisGlobe.controls().autoRotate = true;
		thisGlobe.controls().autoRotateSpeed = data.rotate.speed;
	}
}

/**
 * Handles click actions to display content
 */
function displayContent( thisGlobe, GlobeData, data, position, scroll ) {

	// we could try to use thisGlobe to rotate the globe to the clicked coordinates
	// and we could use position to target specific container, if multiple displayContent were selected

	var contentContainer = document.getElementById('itt_globe_content_' + position + '_' + GlobeData.id );

	if ( contentContainer === undefined || contentContainer === null ) {
		return;
	}
	var thisContent = data.content ? data.content : '';

	contentContainer.innerHTML = thisContent;

	if (scroll) {
		var originalTop = Math.floor(contentContainer.getBoundingClientRect().top - 100);
		window.scrollBy({
			top: originalTop,
			left: 0,
			behavior: 'smooth',
		});
	}

}

function contentBelowScroll( thisGlobe, GlobeData, data ) {
	displayContent( thisGlobe, GlobeData, data, 'below', true );
}

function contentBelow( thisGlobe, GlobeData, data ) {
	displayContent( thisGlobe, GlobeData, data, 'below', false );
}

function contentAbove( thisGlobe, GlobeData, data ) {
	displayContent( thisGlobe, GlobeData, data, 'above', false );
}

function contentRight( thisGlobe, GlobeData, data ) {
	displayContent( thisGlobe, GlobeData, data, 'right', false );
}

function contentLeft( thisGlobe, GlobeData, data ) {
	displayContent( thisGlobe, GlobeData, data, 'left', false );
}

function contentLighbox( thisGlobe, GlobeData, data ) {
	openLightbox( thisGlobe, GlobeData, data, 'inline' );
}

function iframeLightbox( thisGlobe, GlobeData, data ) {
	openLightbox( thisGlobe, GlobeData, data, 'iframe' );
}

function openLightbox( thisGlobe, GlobeData, data, type ) {
	//function (id, data, type) {
	var elements = [],
		width = window.innerWidth < 900 ? '90%' : '60%',
		opts = {}; 

	if (type === 'inline') {

		elements.push({
			//href: data.content,
			content: data.content,
			type: type,
			width: width,
			height: 'auto',
		});
	} else if (type === 'iframe') {
		
		elements.push({
			href: data.content,
			type: 'external',
			width: width,
			height: parseInt(window.innerHeight * 0.8),
		});
	}

	opts = {
		touchNavigation: false,
		draggable: false,
		keyboardNavigation: false,
		loopAtEnd: false,
		loop: false,
		zoomable: false,
		elements: elements,
		closeButton: true, 
		closeOnOutsideClick: true,
	};

	// fix for lightbox closing on bigger touch devices
	/*
	if (window.innerWidth > 768 && functionToCheckifTouchScreenDevice()) {
		opts.closeOnOutsideClick = false;
	}
	*/

	// let's make it global for better debugging, just in case
	ittGlobes.lightbox = GLightbox(opts);

	if (data.content !== '' && ittGlobes.lightbox) {
		ittGlobes.lightbox.open();
		ittGlobes.lightboxIsRunning = true;
	} else {
		console.log('Empty Action Content or Incorrect Request - Lightbox not triggered');
	}

	ittGlobes.lightbox.on('close', function () {
		console.log('lightbox closed, we might need to reset stuff');
		ittGlobes.lightboxIsRunning = false;
	});

	// add custom close button to solve issues on touch devices
	ittGlobes.lightbox.on('open', function () {
		console.log('Open event not working?');
		/*
		let close = document.querySelector('.ginner-container .gslide-media .ittglobe_close');
		if (!close) {
		close = document.createElement('span');
		close.classList.add('ittglobe_close');
		close.innerHTML = 'XXX╳';
		let containers = document.querySelectorAll('.ginner-container .gslide-media');
		containers.forEach(function (el) {
			let clone = close.cloneNode(true);
			clone.addEventListener('click', function () {
			lightbox.close();
			});

			el.prepend(clone);
		});
		}
		*/
	});
}

ittGlobes$1.clickActions.DisplayContentBelow = contentBelow;
ittGlobes$1.clickActions.DisplayContentBelowScroll = contentBelowScroll;
ittGlobes$1.clickActions.DisplayContentAbove = contentAbove;
ittGlobes$1.clickActions.DisplayContentRight = contentRight;
ittGlobes$1.clickActions.DisplayContentLeft = contentLeft;
ittGlobes$1.clickActions.DisplayContentLightbox = contentLighbox;
ittGlobes$1.clickActions.DisplayIframeLightbox = iframeLightbox;

/** initialize globe */
ittGlobes$1.pro = function () {

	// check for globe container created by the shortcode
	let globe_render_container = document.getElementsByClassName( 'js-itt-globe-render' );
	for (const globe_container of globe_render_container) {
		if ( ! globe_container.dataset.hasOwnProperty('globe_id') ) {
			continue;
		}
		let globe_id = 0;
		try {
			globe_id = parseInt( JSON.parse(globe_container.dataset.globe_id ) );
		} catch (e) {
			continue;
		}

		if ( ! globe_container.dataset.hasOwnProperty( 'globe_meta' ) ) {
			continue;
		}

		let globe_meta = {};
		try {
			globe_meta = JSON.parse( globe_container.dataset.globe_meta );
		} catch (e) {
			globe_meta = {};
		}

		extend_globe( globe_id, globe_meta );
	}
};

var extend_globe = function ( id, data ) {
	//data types - normal
	let dataTypes = {
		'arcLines': 'arcLineDefaults',
		'regions': 'regionDefaults',
		'html': 'htmlDefaults',
		'objects': 'pointDefaults',
	};

	globe_ut_prepareProData( data );
	globe_ut_setupData( data, dataTypes );

	let thisGlobe = ittGlobes$1.globesIndex[ id ];

	if ( data.globeImage === 'earth-hollow' ) {
		thisGlobe.showGlobe(false).showAtmosphere(false);
	}

	if( data.globeImage === 'noImage' ){
		// return compatible color object
		let globeColor = typeof data.globeColor !== 'undefined' ? data.globeColor : '#064273';
		thisGlobe.globeMaterial().color.set( globeColor );
	}

	if ( data.globeImage === 'customImage' ) {
		let globeCustomImage = typeof data.customImage !== 'undefined' ?  data.customImage.url : false;
		thisGlobe.globeImageUrl( globeCustomImage );
	}

	addAutoRotate( thisGlobe, data );
	addArcsLayer( thisGlobe, data );
	addHTMLElementsLayer( thisGlobe, data );

	// if there's a live filter, add that info to the main globe
	if( data.layers && Array.isArray(data.layers) && data.liveFilter && data.liveFilter.enabled === "1" ){
		thisGlobe.liveFilter = data.liveFilter;
	}

	// setup marker objects
	if ( data.objects ) {
		addObjectsLayer( thisGlobe, data );
	}

	// overlay layers for Regions, async
	// Wait for the addRegionLayer async operation to complete
	(async () => {
		await Promise.resolve(addRegionLayer(thisGlobe, data));

		// Now that addRegionLayer is complete, we can proceed with the overlay layers
		if (data.layers && Array.isArray(data.layers)) {
			let altitudeMultiplier = 1;
			for (let layer of data.layers) {
				if (layer.regions && Array.isArray(layer.regions)) {

					globe_ut_setupData( layer, {'regions': 'regionDefaults'} );

					// fetch regions
					const existingRegions = thisGlobe.originalPolygonsData || [];
					const preparedLayerRegions = await prepareRegions(thisGlobe, layer, altitudeMultiplier);
					const mergedRegions = [...existingRegions, ...preparedLayerRegions];

					altitudeMultiplier++;

					thisGlobe.originalPolygonsData = mergedRegions;

					if (data.liveFilter && data.liveFilter.enabled === "1" && parseInt(data.liveFilter.default) !== parseInt(id)) {
						if (parseInt(data.liveFilter.default) === parseInt(layer.id)) {
							// we need to keep base, so merge base also
							if (data.liveFilter.keepBase === "1") {
								let baseAndLayerRegions = [...thisGlobe.basePolygonsData, ...preparedLayerRegions];
								thisGlobe.polygonsData(baseAndLayerRegions);
							} else {
								thisGlobe.polygonsData(layer.regions);
							}
						}
					} else {
						thisGlobe.polygonsData(mergedRegions);
					}
				}
			}
		}
	})();

	// overlay layers for dotLabels, points, object markers and arcs
	if( data.layers && Array.isArray( data.layers ) ){
		// Initialize arrays for data collection
		thisGlobe.originalPointsData = thisGlobe.pointsData();
		thisGlobe.originalLabelsData = thisGlobe.labelsData();
		thisGlobe.originalArcsData = thisGlobe.arcsData();
		thisGlobe.originalHtmlData = thisGlobe.htmlElementsData();
		thisGlobe.originalObjectsData = thisGlobe.objectsData();

		// store original base map data, if needed
		thisGlobe.basePointsData = thisGlobe.pointsData();
		thisGlobe.baseLabelsData = thisGlobe.labelsData();
		thisGlobe.baseArcsData = thisGlobe.arcsData();
		thisGlobe.baseHtmlData = thisGlobe.htmlElementsData();
		thisGlobe.baseObjectsData = thisGlobe.objectsData();

		for( let layer of data.layers ){

			let layerDataTypes = {
				'arcLines': 'arcLineDefaults',
				'points': 'pointDefaults',
				'objects': 'pointDefaults',
				'dotLabels': 'labelDefaults',
				'html': 'htmlDefaults',
				'pins': 'pointDefaults'
			};

			globe_ut_setupData( layer, layerDataTypes );
			// points
			if (layer.points && Array.isArray(layer.points)) {

				const existingPoints = thisGlobe.originalPointsData;
				const mergedPoints = [...existingPoints, ...layer.points];

				thisGlobe.originalPointsData = mergedPoints;

				if( data.liveFilter && data.liveFilter.enabled === "1" && parseInt( data.liveFilter.default ) !== parseInt( id ) ){
					if( parseInt( data.liveFilter.default ) === parseInt( layer.id ) ){
						thisGlobe.pointsData(layer.points);
						// we need to keep base, so merge base also
						if( data.liveFilter.keepBase === "1" ){
							let baseAndLayerPoints = [...thisGlobe.basePointsData, ...layer.points];
							thisGlobe.pointsData( baseAndLayerPoints );
						}
					}
				} else {
					thisGlobe.pointsData(mergedPoints);
				}
			}

			// html
			if (layer.html && Array.isArray(layer.html)) {
				const existingHtml = thisGlobe.originalHtmlData;
				const mergedHtml = [...existingHtml, ...layer.html];

				thisGlobe.originalHtmlData = mergedHtml;

				if( data.liveFilter && data.liveFilter.enabled === "1" && parseInt( data.liveFilter.default ) !== parseInt( id ) ){
					if( parseInt( data.liveFilter.default ) === parseInt( layer.id ) ){
						thisGlobe.htmlElementsData(layer.html);
					}
				} else {
					thisGlobe.htmlElementsData(mergedHtml);
				}
			}
			// objects
			if (layer.objects && Array.isArray(layer.objects)) {

				const existingObjects = thisGlobe.originalObjectsData;
				const mergedObjects = [...existingObjects, ...layer.objects];

				thisGlobe.originalObjectsData = mergedObjects;

				if( data.liveFilter && data.liveFilter.enabled === "1" && parseInt( data.liveFilter.default ) !== parseInt( id ) ){
					if( parseInt( data.liveFilter.default ) === parseInt( layer.id ) ){
						thisGlobe.objectsData(layer.objects);
						// we need to keep base, so merge base also
						if( data.liveFilter.keepBase === "1" ){
							let baseAndLayerObjects = [...thisGlobe.baseObjectsData, ...layer.objects];
							thisGlobe.objectsData( baseAndLayerObjects );
						}
					}
				} else {
					thisGlobe.objectsData(mergedObjects);
				}
			}

			// dot labels
			if (layer.dotLabels && Array.isArray(layer.dotLabels)) {
				const existingLabels = thisGlobe.originalLabelsData;
				const mergedLabels = [...existingLabels, ...layer.dotLabels];

				thisGlobe.originalLabelsData = mergedLabels;

				if( data.liveFilter && data.liveFilter.enabled === "1" && parseInt( data.liveFilter.default ) !== parseInt( id ) ){
					if( parseInt( data.liveFilter.default ) === parseInt( layer.id ) ){
						thisGlobe.labelsData(layer.dotLabels);
						// we need to keep base, so merge base also
						if( data.liveFilter.keepBase === "1" ){
							let baseAndLayerDotLabels = [...thisGlobe.baseLabelsData, ...layer.dotLabels];
							thisGlobe.labelsData( baseAndLayerDotLabels );
						}
					}
				} else {
					thisGlobe.labelsData(mergedLabels);
				}
			}

			// arcs
			if (layer.arcLines && Array.isArray(layer.arcLines)) {
				const existingArcs = thisGlobe.originalArcsData;
				const mergedArcs = [...existingArcs, ...layer.arcLines];

				thisGlobe.originalArcsData = mergedArcs;

				if( data.liveFilter && data.liveFilter.enabled === "1" && parseInt( data.liveFilter.default ) !== parseInt( id ) ){
					if( parseInt( data.liveFilter.default ) === parseInt( layer.id ) ){
						thisGlobe.arcsData(layer.arcLines);
						// we need to keep base, so merge base also
						if( data.liveFilter.keepBase === "1" ){
							let baseAndLayerArcs = [...thisGlobe.baseArcsData, ...layer.arcLines];
							thisGlobe.arcsData( baseAndLayerArcs );
						}
					}
				} else {
					thisGlobe.arcsData(mergedArcs);
				}
			}

			// regions are done async
		}
	}

	// if admin
	if ( ittGlobeData.isAdmin ) {

		thisGlobe.onGlobeReady(function () {
			const event = new Event("adminProGlobeReady");
			document.dispatchEvent(event);
		});
	}

	// to make it publically accessible
	thisGlobe.meta = data;
};

ittGlobes$1.pro();
